/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  DMA.h
  \brief	  This is the DMA header File
  \brief	  Configures the DMA HW providing several instances.
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 139 $
  \date  	  $Date: 2011-01-27 16:22:55 -0600 (Thu, 27 Jan 2011) $
  
  * History:  20/August/2008 - Initial Version (IM)
  *   		  12/May/2009 - MISRA Check, Release (IM)
  			  15/Aug/2010 - Added new typedef for callback for context switching (IM)
  			  27/Jan/2011 - Added request support: not only Always request but other peripherals request	


* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/
 

#ifndef _DMA_H_
#define _DMA_H_     

#include	"Common.h"
#include 	"DMA.h"
#include 	"DMAMUX.h"

/* Channel Status definitions */
#define     DMA_CH_IDLE           (0u)
#define     DMA_CH_PREPARED       (1u)
#define     DMA_CH_READY          (2u)
#define     DMA_CH_RUNNING        (3u)
#define     DMA_CH_RUNNING_LITE	  (4u)
#define     DMA_CH_READY_LITE     (5u)

#define     DMA_CH_READY_LSBFVS			(6u)
#define     DMA_CH_READY_LSBFVS_LITE	(7u)

/* DMA MUX request corresponding to Always Enabled <range inclusive> */
/* This special type of request is triggered differently */
#define     DMA_REQ_ALWAYS_START	(62)
#define     DMA_REQ_ALWAYS_END		(62+2)

typedef  enum
{
        DMA_ERROR_OK, 
        DMA_ERROR_BUSY, 
        DMA_ERROR_RANGE, 
        DMA_ERROR_UNINIT, 
        DMA_ERROR_CHANNEL_UNINIT
} DMA_ErrorType;


/* DEPRECATED, will be removed in future releases */
#ifndef REMOVE_DEPRECATED
typedef  struct
{
    uint32_t SourceAddress;
    uint32_t DestinationAddress;
    int16_t  SourceOffset;
    int16_t  DestinationOffset;
    uint32_t MinorLoopCount;
    uint16_t MajorLoopCount;
    int32_t  LastSourceAdjustment;
    int32_t  LastDestinationAdjustment;
    uint8_t  MinorLoopOffsetDest;
    uint8_t  MinorLoopOffsetSrc;
    int32_t  MinorLoopOffset;
} DMA_ChannelCfgType;
#endif

typedef  void ( *DMA_CallbackType )( uint8_t Channel );
#ifdef DCU_LITE
typedef  void ( *DMA_ContextCallbackType )( void );
#endif

/* Configuration Parameters */

#define     DMA_ERRORCHECKER     0	/* < 0 = Checker Off and 1 = Checker ON > */

/* Public Functions */

void          DMA_Init( void );
void          DMA_UnInit( void );
DMA_ErrorType DMA_InitChannel( uint8_t Channel );
void          DMA_UnInitChannel( uint8_t Channel );
#ifndef REMOVE_DEPRECATED
DMA_ErrorType DMA_Configure( uint8_t Channel, DMA_ChannelCfgType *Configuration );
#endif
DMA_ErrorType DMA_SetCallback( uint8_t Channel, DMA_CallbackType Callback );
DMA_ErrorType DMA_EnableRequest( uint8_t Channel, uint8_t Source );
DMA_ErrorType DMA_Start( uint8_t Channel );
DMA_ErrorType DMA_SetReady( uint8_t Channel );
DMA_ErrorType DMA_SetReady_LSBFVS( uint8_t Channel );

#ifdef DCU_LITE
void DMA_SetDCUContextCallback(DMA_ContextCallbackType callback);
void DMA_SetDCULITEContextCallback(DMA_ContextCallbackType callback);
#endif

/* Public Macros Function-alike */

#define     DMA_Done( Channel )                   (EDMA.TCD [ ( Channel ) ].DONE)
#define     DMA_SWStart( Channel )                (EDMA.SSRT.R = ( Channel ))
#define     DMA_Stop( Channel )                   (EDMA.CERQR.R = ( Channel ))
#define     DMA_SourceMod( Channel )           	  (EDMA.TCD [ ( Channel ) ].SMOD)
#define     DMA_BWC( Channel )           	  	  (EDMA.TCD [ ( Channel ) ].BWC)
#define     DMA_SourceAddress( Channel )          (EDMA.TCD [ ( Channel ) ].SADDR)
#define     DMA_DestinationAddress( Channel )     (EDMA.TCD [ ( Channel ) ].DADDR)
#define     DMA_SourceOffset( Channel )           (EDMA.TCD [ ( Channel ) ].SOFF)
#define     DMA_DestinationOffset( Channel )      (EDMA.TCD [ ( Channel ) ].DOFF)
#define     DMA_MinorLoopCount( Channel )         (EDMA.TCD [ ( Channel ) ].NBYTESu.R)
#define 	DMA_GetTransferSize( Channel )		  (EDMA.TCD[ (Channel) ].NBYTESu.R * EDMA.TCD[ (Channel) ].CITER)
#define 	DMA_GetTransferSizeMoff( Channel )	  (EDMA.TCD[ (Channel) ].NBYTESu.B.NBYTES * EDMA.TCD[ (Channel) ].CITER)
#define     DMA_GetMoffBits( Channel )            (EDMA.TCD [ ( Channel ) ].NBYTESu.R&0xC0000000)
#define     DMA_MdestOffsetBit( Channel )  		  (EDMA.TCD[ Channel ].NBYTESu.B.DMLOE)
#define     DMA_MsrcOffsetBit( Channel )  		  (EDMA.TCD[ Channel ].NBYTESu.B.SMLOE)
#define     DMA_MloopOffsetValue( Channel )  	  (EDMA.TCD[ Channel ].NBYTESu.B.MLOFF)
/* Violation to [MISRA 19.4] Cannot brace the whole expression because it has two lines */
#define     DMA_SetMajorLoopCount( Channel, Value )		EDMA.TCD[ ( Channel ) ].BITER = ( Value );	\
                										EDMA.TCD[ (Channel) ].CITER = (Value)
                										
#define     DMA_SetMajorLoopCountLink(Channel, Link, Count)	EDMA.TCD[(Channel)].BITERE_LINK = 1;	\
															EDMA.TCD[(Channel)].CITERE_LINK = 1;	\
															EDMA.TCD[(Channel)].BITER = ((Link<<9)|(Count));	\
                											EDMA.TCD[(Channel)].CITER = ((Link<<9)|(Count))
                										
                										
#define     DMA_LastSourceAdjustment( Channel )          (EDMA.TCD [ ( Channel ) ].SLAST)
#define     DMA_LastDestinationAdjustment( Channel )     (EDMA.TCD [ ( Channel ) ].DLAST_SGA)

#define     DMA_SourceSize( Channel )                    (EDMA.TCD [ ( Channel ) ].SSIZE)
#define     DMA_DestinationSize( Channel )               (EDMA.TCD [ ( Channel ) ].DSIZE)
#define     DMA_isBusy( )                                ( DMA_BusyFlag )
#define     DMA_UnLock( )                                ( DMA_BusyFlag = 0u)
#define     DMA_getChannelStatus( Channel )              ( DMA_ChannelStatus [ ( Channel ) ] )


/* End of user modifiable parameters */


/********************** Private Data, DO NOT MODIFY ***************************************************************/
/**************************** DO NOT MODIFY ***********************************************************************/
/**************************** DO NOT MODIFY ***********************************************************************/

#define     DMA_MAX_DMACHANNELS                           16u

extern  uint8_t DMA_BusyFlag;
extern  uint8_t DMA_ChannelStatus[ DMA_MAX_DMACHANNELS ];

/* Interrupt Prototypes */
void DMA_CH0_ISR( void );
void DMA_CH1_ISR( void );
void DMA_CH2_ISR( void );
void DMA_CH3_ISR( void );
void DMA_CH4_ISR( void );
void DMA_CH5_ISR( void );
void DMA_CH6_ISR( void );
void DMA_CH7_ISR( void );
void DMA_CH8_ISR( void );
void DMA_CH9_ISR( void );
void DMA_CH10_ISR( void );
void DMA_CH11_ISR( void );
void DMA_CH12_ISR( void );
void DMA_CH13_ISR( void );
void DMA_CH14_ISR( void );
void DMA_CH15_ISR( void );

#endif
